using System;
using System.Collections.Generic;
using System.Text;
using System.Xml;
using System.Xml.Serialization;
using System.Windows.Forms;
using NWN2Toolset.NWN2.Data;
using NWN2Toolset.NWN2.Data.TypedCollections;
using NWN2Toolset.NWN2.Data.ConversationData;
using NWN2Toolset.NWN2.Views;
using GlacialComponents.Controls.GlacialTreeList;
using NWN2Toolset.NWN2.Data.Templates;
using OEIShared.Utils;
using ManagedElectron;
using OEIShared.IO;
using NWN2Toolset;
using OEIShared.IO.TwoDA;

namespace XmlExporter
{
    [Serializable]
    public class ContentNode
    {
        [XmlAttribute]
        public int idNum = -1;

        [XmlAttribute]
        public int orderNum;

        [XmlElement]
        public string conversationText;

        [XmlElement]
        public string conversationComments;

        [XmlArrayItem("ContentNode")]
        public List<ContentNode> subNodes = new List<ContentNode>();

        [XmlAttribute]
        public int linkTo;

        [XmlArrayItem("Info")]
        public List<Info> additionalData = new List<Info>();

        [XmlArrayItem("InfoObject")]
        public List<InfoObject> infoObjects = new List<InfoObject>();

        [XmlAttribute(AttributeName = "nodeType")]
        public string nodeTypeString
        {
            get
            {
                if (myNodeTypeEnum == ConversationNodeType.NPC)
                {
                    return "npc";
                }
                else if (myNodeTypeEnum == ConversationNodeType.PC)
                {
                    return "pc";
                }
                else
                {
                    return "";
                }
            }
            set
            {
                if (value == "pc")
                {
                    myNodeTypeEnum = ConversationNodeType.PC;
                }
                else if (value == "npc")
                {
                    myNodeTypeEnum = ConversationNodeType.NPC;
                }
                else
                {
                    throw new Exception("This GUI node does not have a proper nodeType or is a root node and should not have a content node.");
                }
            }
        }

        private ConversationNodeType myNodeTypeEnum;

        [XmlIgnore]
        public ConversationNodeType nodeType
        {
            get
            {
                return myNodeTypeEnum;
            }
            set
            {
                myNodeTypeEnum = value;
            }
        }

        [XmlIgnore]
        public string myGuid;

        [XmlIgnore]
        public string myConnectorID = "";

        [XmlIgnore]
        public bool isLink
        {
            get
            {
                return (linkTo > 0);
            }
        }

        [XmlIgnore]
        public Conversation myConversation;

        [XmlIgnore]
        public NWN2ConversationConnector myConnector;
        
        public static ContentNode NewContentNode(int nextIdNum, int nextOrderNum)
        {
            ContentNode newNode = new ContentNode();
            newNode.idNum = nextIdNum;
            newNode.orderNum = nextOrderNum;
            return newNode;
        }

        public static ContentNode NewContentNodeLink(int nextOrderNum)
        {
            ContentNode newNode = new ContentNode();
            newNode.orderNum = nextOrderNum;
            return newNode;
        }

        public ContentNode SearchContentNodeById(int checkIdNum)
        {
            ContentNode tempNode = null;
            if (idNum == checkIdNum)
            {
                return this;
            }
            foreach (ContentNode subNode in subNodes)
            {
                tempNode = subNode.SearchContentNodeById(checkIdNum);
                if (tempNode != null)
                {
                    return tempNode;
                }
            }
            return null;
        }

        public ContentNode DuplicateContentNode(int nextIdNum, int nextOrderNum)
        {
            ContentNode newNode = new ContentNode();
            newNode.conversationComments = this.conversationComments;
            newNode.conversationText = this.conversationText;
            newNode.idNum = nextIdNum;
            newNode.linkTo = this.linkTo;
            newNode.nodeType = this.nodeType;
            newNode.orderNum = nextOrderNum;
            return newNode;
        }

        public ContentNode Duplicate()
        {
            ContentNode copy = new ContentNode();
            
            copy.additionalData = this.additionalData;
            copy.conversationComments = this.conversationComments;
            copy.conversationText = this.conversationText;
            copy.idNum = this.idNum;
            copy.linkTo = this.linkTo;
            copy.nodeType = this.nodeType;
            copy.orderNum = this.orderNum;
            foreach (ContentNode node in this.subNodes)
            {
                copy.subNodes.Add(node.Duplicate());
            }
            return copy;
        }

        public static ConversationNodeType GetOppositeType(ConversationNodeType type)
        {
            if (type == ConversationNodeType.NPC)
            {
                return ConversationNodeType.PC;
            }
            else
            {
                return ConversationNodeType.NPC;
            }
        }

        public static InfoObject SetActions(NWN2ScriptFunctorCollection actions)
        {
            InfoObject infoActions = new InfoObject("Actions");
            InfoObject eachAction;
            foreach (NWN2ScriptFunctor script in actions)
            {
                if (script.Script != null)
                {
                    if (script.Script.ResRef != null)
                    {
                        eachAction = new InfoObject(script.Script.ResRef.ToString());
                        foreach (NWN2ScriptParameter parameter in script.Parameters)
                        {
                            if (parameter.ParameterType == NWN2ScriptParameterType.Float)
                            {
                                eachAction.AddItem("Float", parameter.ValueFloat.ToString());
                            }
                            else if (parameter.ParameterType == NWN2ScriptParameterType.Int)
                            {
                                eachAction.AddItem("Int", parameter.ValueInt.ToString());
                            }
                            else if (parameter.ParameterType == NWN2ScriptParameterType.String)
                            {
                                eachAction.AddItem("String", parameter.ValueString);
                            }
                            else if (parameter.ParameterType == NWN2ScriptParameterType.Tag)
                            {
                                eachAction.AddItem("Tag", parameter.ValueTag);
                            }
                        }
                        infoActions.AddSubObject(eachAction);
                    }
                }
            }

            return infoActions;
        }

        public static InfoObject SetCameraInfo(NWN2ConversationCameraInfo cameraInfo)
        {
            InfoObject camera = new InfoObject("CameraInfo");
            camera.AddItem("Mode", cameraInfo.Mode.ToString());
            camera.AddItem("MovementType", cameraInfo.MovementType.ToString());
            camera.AddItem("Offset1", SetOffset(cameraInfo.Offset1));
            camera.AddItem("Offset2", SetOffset(cameraInfo.Offset2));
            camera.AddItem("Orientation", SetOrientation(cameraInfo.Orientation));
            camera.AddItem("ShotType", cameraInfo.ShotType.ToString());
            camera.AddItem("StaticCamera", cameraInfo.StaticCamera);

            return camera;
        }

        public static string SetOffset(Microsoft.DirectX.Vector3 offset)
        {
            string stringOffset = "";
            stringOffset = offset.X + "," + offset.Y + "," + offset.Z;
            return stringOffset;
        }

        public static string SetOrientation(OEIShared.OEIMath.RHQuaternion orientation)
        {
            string stringOrientation = "";
            stringOrientation = orientation.X + "," + orientation.Y + "," + orientation.Z + "," + orientation.W;
            return stringOrientation;
        }

        public static InfoObject SetConditions(NWN2ConditionalFunctorCollection conditions, Log log)
        {
            InfoObject infoConditions = new InfoObject("Conditions");
            InfoObject eachCondition;
            if (conditions != null)
            {
                log.WriteLogLine("Conditions not null.");
                foreach (NWN2ConditionalFunctor condition in conditions)
                {
                    if (condition.Script != null)
                    {
                        if (condition.Script.ResRef != null)
                        {
                            eachCondition = new InfoObject(condition.Script.ResRef.ToString());
                            log.WriteLogLine("Exporting a condition.");
                            log.WriteLogLine("Added condition.Script.FullName.");

                            eachCondition.AddItem("NOT", condition.Not.ToString());

                            foreach (NWN2ScriptParameter parameter in condition.Parameters)
                            {
                                log.WriteLogLine("Starting parameters.");
                                if (parameter.ParameterType == NWN2ScriptParameterType.Float)
                                {
                                    log.WriteLogLine("Parameter is Float.");
                                    eachCondition.AddItem("Float", parameter.ValueFloat.ToString());
                                }
                                else if (parameter.ParameterType == NWN2ScriptParameterType.Int)
                                {
                                    log.WriteLogLine("Parameter is Int.");
                                    eachCondition.AddItem("Int", parameter.ValueInt.ToString());
                                }
                                else if (parameter.ParameterType == NWN2ScriptParameterType.String)
                                {
                                    log.WriteLogLine("Parameter is String.");
                                    eachCondition.AddItem("String", parameter.ValueString);
                                }
                                else if (parameter.ParameterType == NWN2ScriptParameterType.Tag)
                                {
                                    log.WriteLogLine("Parameter is other type.");
                                    eachCondition.AddItem("Tag", parameter.ValueTag);
                                }
                            }
                            log.WriteLogLine("Finished adding parameters for this condition.");
                            infoConditions.AddSubObject(eachCondition);
                            log.WriteLogLine("Added this condition to collection.");
                        }
                    }
                }
                log.WriteLogLine("Finished adding conditions.");
            }
            return infoConditions;
        }

        public static string SetNodeText(OEIExoLocString text)
        {
            string textString;
            textString = text.ToString();
            textString = textString.Replace(Environment.NewLine, "");
            if (text != null)
            {
                if (text.Strings != null)
                {
                    if (text.Strings.Count > 0)
                    {
                        if (text.Strings[0].ToString() != "")
                        {
                            textString = text.Strings[0].ToString();
                            textString = textString.Replace(Environment.NewLine, "");
                        }
                    }
                }
            }
            if (textString == "\"\", ")
            {
                textString = "";
            }
            return textString;
        }

        public static string SetSound(IResourceEntry sound)
        {
            if (sound != null)
            {
                if (sound.ResRef != null)
                {
                    return sound.ResRef.Value;
                }
            }
            return "";
        }

        public static string SetTGAToDisplay(IResourceEntry tgaToDisplay)
        {
            if (tgaToDisplay != null)
            {
                if (tgaToDisplay.ResRef != null)
                {
                    return tgaToDisplay.ResRef.Value;
                }
            }
            return "";
        }

        public static InfoObject SetSpeakerAppearance(TwoDAReference speaker)
        {
            if (speaker == null)
            {
                return null;
            }
            InfoObject infoSpeakerAppearance = new InfoObject("SpeakerAppearance");
            infoSpeakerAppearance.AddItem("TwoDAName", speaker.TwoDAName);
            infoSpeakerAppearance.AddItem("ColumnName", speaker.ColumnName);
            infoSpeakerAppearance.AddItem("IsStringRef", speaker.IsStringRef.ToString());
            infoSpeakerAppearance.AddItem("Row", speaker.Row.ToString());

            return infoSpeakerAppearance;
        }

        public static ContentNode NodeFromNWN2Connector(NWN2ConversationConnector connector, int orderNum, ConversationNodeType type, Conversation conversation)
        {
            Log log = conversation.log;
            log.WriteLogLine("Starting export function");

            ConversationNodeType oppositeType = GetOppositeType(type);
            log.WriteLogLine("Set oppositetype variable");

            ContentNode newNode = new ContentNode();
            newNode.myConnector = connector;
            log.WriteLogLine("Created new node");
            
            newNode.myConversation = conversation;
            log.WriteLogLine("Exported conversation");
            
            newNode.nodeType = type;
            log.WriteLogLine("Exported nodetype");
            
            newNode.orderNum = orderNum;
            log.WriteLogLine("Exported ordernum");
            
            newNode.idNum = conversation.maxIdNum;
            conversation.maxIdNum++;
            log.WriteLogLine("Exported idnum");

            newNode.infoObjects.Add(SetActions(connector.Actions));
            log.WriteLogLine("Exported actions");

            newNode.additionalData.Add(new Info("Animation", connector.Animation.ToString()));
            log.WriteLogLine("Exported animation");
            
            newNode.additionalData.Add(new Info("AutomaticallyCreated", connector.AutomaticallyCreated.ToString()));
            log.WriteLogLine("Exported automatically created");
            
            newNode.infoObjects.Add(SetCameraInfo(connector.CameraInfo));
            log.WriteLogLine("Exported camerainfo");
            
            newNode.conversationComments = connector.Comment;
            log.WriteLogLine("Exported conversationcomments");

            newNode.infoObjects.Add(SetConditions(connector.Conditions, log));
            log.WriteLogLine("Exported conditions");

            newNode.additionalData.Add(new Info("ConnectorID", connector.ConnectorID.ID.ToString()));
            log.WriteLogLine("Exported connectorid");
            
            newNode.additionalData.Add(new Info("Delay", connector.Delay.ToString()));
            log.WriteLogLine("Exported delay");
            
            newNode.additionalData.Add(new Info("Entry", connector.Line.Entry.ToString()));
            log.WriteLogLine("Exported entry");
            
            newNode.additionalData.Add(new Info("LineGuid", connector.Line.LineGuid.ToString()));
            log.WriteLogLine("Exported lineguid");
            
            newNode.myGuid = connector.Line.LineGuid.ToString();
            log.WriteLogLine("Exported newNode.myGuid");

            newNode.additionalData.Add(new Info("Link", connector.Link.ToString()));
            log.WriteLogLine("Exported Link");
            
            newNode.additionalData.Add(new Info("LinkComment", connector.LinkComment));
            log.WriteLogLine("Exported linkcomment");
            
            newNode.additionalData.Add(new Info("Listener", connector.Listener));
            log.WriteLogLine("Exported listener");
            
            newNode.additionalData.Add(new Info("MainQuestionNodeLink", connector.Line.MainQuestionNodeLink.ToString()));
            log.WriteLogLine("Exported mainquestionnodelink");
            
            newNode.additionalData.Add(new Info("NeedsVO", connector.NeedsVO.ToString()));
            log.WriteLogLine("Exported needsvo");
            
            newNode.additionalData.Add(new Info("Quest", connector.Quest.Quest + "-" + connector.Quest.Entry.ToString()));
            log.WriteLogLine("Exported quest");
            
            newNode.additionalData.Add(new Info("ShowOnce", connector.ShowOnce.ToString()));
            log.WriteLogLine("Exported showonce");

            newNode.additionalData.Add(new Info("Sound", SetSound(connector.Sound)));
            log.WriteLogLine("Exported sound");
            
            newNode.additionalData.Add(new Info("Speaker", connector.Speaker));
            log.WriteLogLine("Exported speaker");

            newNode.infoObjects.Add(SetSpeakerAppearance(connector.SpeakerAppearance));
            log.WriteLogLine("Exported speakerappearance");
            
            newNode.additionalData.Add(new Info("SpeakerGender", connector.SpeakerGender.ToString()));
            log.WriteLogLine("Exported speakergender");
            
            newNode.additionalData.Add(new Info("SpeakerHeadVariation", connector.SpeakerHeadVariation.ToString()));
            log.WriteLogLine("Exported speakerheadvariation");

            newNode.conversationText = SetNodeText(connector.Text);
            log.WriteLogLine("Exported text:" + newNode.conversationText);

            newNode.additionalData.Add(new Info("TGAToDisplay", SetTGAToDisplay(connector.TGAToDisplay)));
            log.WriteLogLine("Exported tgatodisplay");
            
            newNode.additionalData.Add(new Info("Type", connector.Type.ToString()));
            log.WriteLogLine("Exported type");
            
            newNode.additionalData.Add(new Info("VONeedsToBeRecorded", connector.VONeedsToBeRecorded.ToString()));
            log.WriteLogLine("Exported voneedstoberecorded");

            if (connector.Line.Children.Count > 0 && !connector.Link)
            {
                int i = 1;
                foreach(NWN2ConversationConnector subConnector in connector.Line.Children)
                {
                    newNode.subNodes.Add(ContentNode.NodeFromNWN2Connector(subConnector, i, oppositeType, conversation));
                    i++;
                }
            }
            return newNode;
        }

        public void ConnectLinks()
        {
            if (myConnector.Link)
            {
                if (myConversation == null)
                    MessageBox.Show("myConversation null on node with text '" + this.conversationText + "'");
                if (myGuid == null)
                    MessageBox.Show("myGuid null on node with text '" + this.conversationText + "'");
                this.additionalData = new List<Info>();
                this.infoObjects = new List<InfoObject>();
                this.subNodes = new List<ContentNode>();
                ContentNode original = myConversation.FindNodeByLineGuid(myGuid);
                if (original == null)
                {
                    MessageBox.Show("A link is corrupted. It will be displayed with the words '[Broken link:]' in the conversation editor.");
                    this.linkTo = 0;
                    this.conversationText = "[Broken link:] " + this.conversationText;
                    this.conversationComments = "";
                    this.idNum = this.myConversation.maxIdNum;
                    this.myConversation.maxIdNum++;
                }
                else
                {
                    this.conversationComments = "";
                    this.conversationText = "";
                    this.idNum = 0;
                    if (original.idNum == -1)
                        MessageBox.Show("original.idNum -1 on node with text '" + this.conversationText + "'");
                    this.linkTo = original.idNum;
                }
            }
            foreach (ContentNode node in subNodes)
            {
                node.ConnectLinks();
            }
        }

        public ContentNode FindNodeByLineGuid(string lineGuid)
        {
            ContentNode tempNode = null;

            if ((myGuid == lineGuid) && !myConnector.Link)
            {
                return this;
            }
            foreach (ContentNode node in subNodes)
            {
                tempNode = node.FindNodeByLineGuid(lineGuid);
                if (tempNode != null)
                {
                    return tempNode;
                }
            }
            return null;
        }

        public string GetInfoValueByName(string name)
        {
            foreach (Info info in additionalData)
            {
                if (info.variableName == name)
                {
                    return info.variableValue;
                }                
            }
            return "";
        }

        public static NWN2ScriptParameter ParseParameter(Info parameter)
        {
            NWN2ScriptParameter tempParameter = new NWN2ScriptParameter();

            float tempFloat;
            int tempInt;

            if (parameter.variableName == "Float")
            {
                tempParameter.ParameterType = NWN2ScriptParameterType.Float;
                if (float.TryParse(parameter.variableValue, out tempFloat))
                {
                    tempParameter.ValueFloat = tempFloat;
                }
            }
            else if (parameter.variableName == "Int")
            {
                tempParameter.ParameterType = NWN2ScriptParameterType.Int;
                if (int.TryParse(parameter.variableValue, out tempInt))
                {
                    tempParameter.ValueInt = tempInt;
                }
            }
            else if (parameter.variableName == "String")
            {
                tempParameter.ParameterType = NWN2ScriptParameterType.String;
                tempParameter.ValueString = parameter.variableValue;
            }
            else if (parameter.variableName == "Tag")
            {
                tempParameter.ParameterType = NWN2ScriptParameterType.Tag;
                tempParameter.ValueTag = parameter.variableValue;
            }
            else
            {
                return null;
            }
            
            return tempParameter;
        }

        public static NWN2ScriptFunctorCollection GetActions(InfoObject actions, Log myLog)
        {
            NWN2ScriptFunctor functor;
            NWN2ScriptParameter tempParameter;
            NWN2ScriptFunctorCollection toReturn = new NWN2ScriptFunctorCollection();

            if (actions.subObjects.Count <= 0)
            {
                return null;
            }
            foreach (InfoObject action in actions.subObjects)
            {
                if (action.objectName != "")
                {
                    functor = new NWN2ScriptFunctor();

                    myLog.WriteLogLine("Script name is " + action.objectName);

                    functor.Script = GetResource(action.objectName, "NSS", myLog);

                    foreach (Info parameter in action.info)
                    {
                        tempParameter = ParseParameter(parameter);
                        if (tempParameter != null)
                        {
                            functor.Parameters.Add(tempParameter);
                            myLog.WriteLogLine("Added parameter " + tempParameter.ToString());
                        }
                    }
                    if (functor.Parameters.Count > 0)
                    {
                        toReturn.Add(functor);
                    }
                }
            }
            if (toReturn.Count > 0)
            {
                return toReturn;
            }
            else
            {
                return null;
            }
        }

        public static NWN2_ConversationAnimType GetAnimation(string animation)
        {
            if (ArrayContains(Enum.GetNames(typeof(NWN2_ConversationAnimType)), animation))
            {
                return (NWN2_ConversationAnimType)Enum.Parse(typeof(NWN2_ConversationAnimType), animation);
            }
            return NWN2_ConversationAnimType.CONVERSATION_ANIMTYPE_IDLE;
        }

        public InfoObject GetInfoObjectByName(string name)
        {
            foreach (InfoObject info in infoObjects)
            {
                if (info.objectName == name)
                    return info;
            }
            return new InfoObject();
        }

        public static NWN2ConversationCameraOverrideMode GetCameraMode(string mode)
        {
            if(ArrayContains(Enum.GetNames(typeof(NWN2ConversationCameraOverrideMode)), mode))
            {
                return (NWN2ConversationCameraOverrideMode)Enum.Parse(typeof(NWN2ConversationCameraOverrideMode), mode);
            }
            return NWN2ConversationCameraOverrideMode.None;
        }

        public static NWN2ConversationCameraMovementType GetCameraMovementType(string movement)
        {
            if (ArrayContains(Enum.GetNames(typeof(NWN2ConversationCameraMovementType)), movement))
            {
                return (NWN2ConversationCameraMovementType)Enum.Parse(typeof(NWN2ConversationCameraMovementType), movement);
            }
            return NWN2ConversationCameraMovementType.None;
        }

        public static Microsoft.DirectX.Vector3 GetCameraOffset(string offset)
        {
            if (offset == "" || offset.IndexOf(",") < 0)
            {
                return new Microsoft.DirectX.Vector3(0f, 0f, 0f);
            }
            
            string x;
            string y;
            string z;

            x = offset.Substring(0, offset.IndexOf(","));
            offset = offset.Substring(offset.IndexOf(",") + 1);
            y = offset.Substring(0, offset.IndexOf(","));
            offset = offset.Substring(offset.IndexOf(",") + 1);
            z = offset;

            if (x != "" && y != "" && z != "")
            {
                return new Microsoft.DirectX.Vector3(float.Parse(x), float.Parse(y), float.Parse(z));
            }
            else
            {
                return new Microsoft.DirectX.Vector3(0f, 0f, 0f);
            }
        }

        public static OEIShared.OEIMath.RHQuaternion GetCameraOrientation(string text)
        {
            if (text == "")
            {
                return new OEIShared.OEIMath.RHQuaternion(0f, 0f, 0f, 0f);
            }

            string x;
            string y;
            string z;
            string w;

            x = text.Substring(0, text.IndexOf(","));
            text = text.Substring(text.IndexOf(",") + 1);
            y = text.Substring(0, text.IndexOf(","));
            text = text.Substring(text.IndexOf(",") + 1);
            z = text.Substring(0, text.IndexOf(","));
            text = text.Substring(text.IndexOf(",") + 1);
            w = text;

            if (x != "" && y != "" && z != "" && w != "")
            {
                return new OEIShared.OEIMath.RHQuaternion(float.Parse(x), float.Parse(y), float.Parse(z), float.Parse(w));
            }
            else
            {
                return new OEIShared.OEIMath.RHQuaternion(0f, 0f, 0f, 0f);
            }
        }

        public static NWN2ConversationCameraShotType GetCameraShotType(string text)
        {
            if (ArrayContains(Enum.GetNames(typeof(NWN2ConversationCameraShotType)), text))
            {
                return (NWN2ConversationCameraShotType)Enum.Parse(typeof(NWN2ConversationCameraShotType), text);
            }
            return NWN2ConversationCameraShotType.CloseUp;
        }

        public static NWN2ConversationCameraInfo GetCameraInfo(InfoObject camera)
        {
            NWN2ConversationCameraInfo camInfo = new NWN2ConversationCameraInfo();
            camInfo.Mode = GetCameraMode(camera.GetInfoByName("Mode"));
            camInfo.MovementType = GetCameraMovementType(camera.GetInfoByName("MovementType"));
            camInfo.Offset1 = GetCameraOffset(camera.GetInfoByName("Offset1"));
            camInfo.Offset2 = GetCameraOffset(camera.GetInfoByName("Offset2"));
            camInfo.Orientation = GetCameraOrientation(camera.GetInfoByName("Orientation"));
            camInfo.ShotType = GetCameraShotType(camera.GetInfoByName("ShotType"));
            camInfo.StaticCamera = camera.GetInfoByName("StaticCamera");

            return camInfo;
        }

        public static IResourceEntry GetResource(string name, string type, Log myLog)
        {
            ushort resourceType = BWResourceTypes.GetResourceType(type);
            IResourceEntry resource = null;
            OEIResRef resourceRef = new OEIResRef(name);
            myLog.WriteLogLine("Looking for resource of type '" + type + "' with name '" + name + "'");
            foreach (ResourceRepository repo in ResourceManager.Instance.Repositories)
            {
                resource = repo.FindResource(resourceRef, resourceType);
                if (resource != null)
                {
                    return resource;
                }
            }
            myLog.WriteLogLine("Script not found in any repositories.");
            return null;
        }

        public static NWN2ConditionalFunctorCollection GetConditions(InfoObject infoConditions, Log myLog)
        {
            NWN2ConditionalFunctorCollection conditions = new NWN2ConditionalFunctorCollection();
            NWN2ConditionalFunctor functor;
            NWN2ScriptParameter tempParameter;

            if (infoConditions.subObjects.Count <= 0)
            {
                return null;
            }

            foreach (InfoObject condition in infoConditions.subObjects)
            {
                if (condition.objectName != "")
                {
                    functor = new NWN2ConditionalFunctor();

                    functor.Script = GetResource(condition.objectName, "NSS", myLog);

                    functor.Not = ConvertStringToBool(condition.GetInfoByName("NOT"));

                    foreach (Info parameter in condition.info)
                    {
                        if (parameter.variableName != "Type")
                        {
                            tempParameter = ParseParameter(parameter);
                            if (tempParameter != null)
                            {
                                functor.Parameters.Add(tempParameter);
                            }
                        }
                    }
                    if (functor.Parameters.Count > 0)
                    {
                        conditions.Add(functor);
                    }
                }
            }
            if (conditions.Count > 0)
            {
                return conditions;
            }
            else
            {
                return null;
            }
        }

        public static NWN2ConversationShowOnceType GetShowOnce(string text)
        {
            if(ArrayContains(Enum.GetNames(typeof(NWN2ConversationShowOnceType)), text))
            {
                return (NWN2ConversationShowOnceType)Enum.Parse(typeof(NWN2ConversationShowOnceType), text);
            }
            return NWN2ConversationShowOnceType.Always;
        }

        public static bool ArrayContains(string[] array, string contains)
        {
            foreach (string content in array)
            {
                if (content == contains)
                {
                    return true;
                }
            }
            return false;
        }

        public static CreatureGender GetSpeakerGender(string text)
        {
            if (ArrayContains(Enum.GetNames(typeof(CreatureGender)), text))
            {
                return (CreatureGender)Enum.Parse(typeof(CreatureGender), text, false);
            }
            return CreatureGender.Male;
        }

        public NWN2ConversationConnectorType GetConnectorType(string text)
        {
            if (text == "")
            {
                if (this.nodeType == ConversationNodeType.PC)
                {
                    return NWN2ConversationConnectorType.Reply;
                }
                else
                {
                    if (this.myConversation.IsSubnode(this))
                    {
                        return NWN2ConversationConnectorType.StartingEntry;
                    }
                    else
                    {
                        return NWN2ConversationConnectorType.Entry;
                    }
                }
            }
            else if (ArrayContains(Enum.GetNames(typeof(NWN2ConversationConnectorType)), text))
            {
                return (NWN2ConversationConnectorType)Enum.Parse(typeof(NWN2ConversationConnectorType), text);
            }
            else
            {
                return GetConnectorType("");
            }
        }

        public static bool ConvertStringToBool(string text)
        {
            if (text == "True" || text == "true")
            {
                return true;
            }
            else
            {
                return false;
            }
        }

        public static NWN2QuestTuple GetQuest(string questString)
        {
            if (questString != "")
            {
                NWN2QuestTuple quest = new NWN2QuestTuple();
                string[] questTuple = questString.Split(new string[1] { "-" }, StringSplitOptions.None);
                quest.Quest = questTuple[0];
                quest.Entry = uint.Parse(questTuple[1]);
                return quest;
            }
            else
            {
                return null;
            }
        }

        public static IResourceEntry GetSound(string soundString, Log myLog)
        {
            if (soundString != "")
            {
                return GetResource(soundString, "WAV", myLog);
            }
            return null;
        }

        public static TwoDAReference GetSpeakerAppearance(InfoObject speaker)
        {
            TwoDAReference appearance;
            if (speaker != null)
            {
                if (speaker.GetInfoByName("TwoDAName") != "" && speaker.GetInfoByName("ColumnName") != "" && speaker.GetInfoByName("IsStringRef") != "" && speaker.GetInfoByName("Row") != "")
                {
                    appearance = new TwoDAReference(speaker.GetInfoByName("TwoDAName"), speaker.GetInfoByName("ColumnName"), ConvertStringToBool(speaker.GetInfoByName("IsStringRef")), int.Parse(speaker.GetInfoByName("Row")));
                }
                else
                    appearance = new TwoDAReference("Appearance", "STRING_REF", true, 0);
            }
            else
            {
                appearance = new TwoDAReference("Appearance", "STRING_REF", true, 0);
            }

            return appearance;
        }

        public static IResourceEntry GetTGAToDisplay(string tgaString, Log myLog)
        {
            IResourceEntry tgaResource = null;
            if (tgaString != "")
            {
                tgaResource = GetResource(tgaString, "TGA", myLog);
            }
            return tgaResource;
        }

        public void AddMyChildrenToMyNWN2ConversationConnector(NWN2GameConversation nwn2Conversation, Log myLog)
        {
            NWN2ConversationConnector tempConnector;
            foreach (ContentNode node in subNodes)
            {
                if (!node.isLink)
                {
                    tempConnector = nwn2Conversation.InsertChild(myConnector);
                    node.PopulateNWN2ConversationConnector(tempConnector, myLog);
                    node.AddMyChildrenToMyNWN2ConversationConnector(nwn2Conversation, myLog);
                }
            }
        }

        public void AddMyChildLinksToMyNWN2ConversationConnector(Conversation conversation, NWN2GameConversation nwn2Conversation, Log myLog)
        {
            NWN2ConversationLine tempLine;
            NWN2ConversationConnector linkConnector;
            int index;
            foreach (ContentNode node in subNodes)
            {
                if (node.isLink)
                {
                    tempLine = conversation.GetNWN2LineByID(node.linkTo);
                    if (tempLine != null)
                    {
                        linkConnector = nwn2Conversation.InsertLink(this.myConnector, tempLine);
                        index = linkConnector.Parent.Line.Children.IndexOf(linkConnector);
                        while (index >= node.orderNum)
                        {
                            nwn2Conversation.MoveUp(linkConnector);
                            index = linkConnector.Parent.Line.Children.IndexOf(linkConnector);
                        }
                    }
                }
                else
                {
                    node.AddMyChildLinksToMyNWN2ConversationConnector(conversation, nwn2Conversation, myLog);
                }
            }
        }

        public void PopulateNWN2ConversationConnector(NWN2ConversationConnector newCon, Log myLog)
        {
            myLog.WriteLogLine("Starting populate function");
            if (this.isLink)
            {
                throw new Exception("Cannot populate connector with link at this stage.");
            }

            this.myConnector = newCon;
            myLog.WriteLogLine("Creating line: " + this.conversationText);
            
            NWN2ScriptFunctorCollection actions = GetActions(GetInfoObjectByName("Actions"), myLog);
            if (actions != null)
            {
                newCon.Actions = actions;
            }
            myLog.WriteLogLine("Imported Actions.");

            newCon.Animation = GetAnimation(GetInfoValueByName("Animation"));
            myLog.WriteLogLine("Imported Animation.");

            newCon.AutomaticallyCreated = ConvertStringToBool(GetInfoValueByName("AutomaticallyCreated"));
            myLog.WriteLogLine("Imported AutomaticallyCreated.");

            newCon.CameraInfo = GetCameraInfo(GetInfoObjectByName("CameraInfo"));
            myLog.WriteLogLine("Imported CameraInfo.");

            newCon.Comment = conversationComments;
            myLog.WriteLogLine("Imported Comment.");

            NWN2ConditionalFunctorCollection conditions = GetConditions(GetInfoObjectByName("Conditions"), myLog);
            if (conditions != null)
            {
                newCon.Conditions = conditions;
            }
            myLog.WriteLogLine("Imported Conditions.");

            this.myConnectorID = newCon.ConnectorID.ToString();
            myLog.WriteLogLine("Set myConnectorID variable.");

            if (GetInfoValueByName("Delay") != "")
            {
                newCon.Delay = float.Parse(GetInfoValueByName("Delay"));
            }
            myLog.WriteLogLine("Imported Delay.");

            newCon.Link = false;
            myLog.WriteLogLine("Imported Link.");

            newCon.LinkComment = GetInfoValueByName("LinkComment");
            myLog.WriteLogLine("Imported LinkComment.");

            newCon.Listener = GetInfoValueByName("Listener");
            myLog.WriteLogLine("Imported Listener.");

            string mqnGuid = GetInfoValueByName("MainQuestionNodeLink");
            if (mqnGuid != "")
            {
                newCon.Line.MainQuestionNodeLink = new Guid(mqnGuid);
            }
            else
            {
                newCon.Line.MainQuestionNodeLink = new Guid("00000000-0000-0000-0000-000000000000");
            }
            myLog.WriteLogLine("Imported MainQuestionNodeLink");

            newCon.NeedsVO = ConvertStringToBool(GetInfoValueByName("NeedsVO"));
            myLog.WriteLogLine("Imported NeedsVO.");

            NWN2QuestTuple questTuple = GetQuest(GetInfoValueByName("Quest"));
            if (questTuple != null)
            {
                newCon.Quest = questTuple;
            }
            myLog.WriteLogLine("Imported Quest.");

            newCon.ShowOnce = GetShowOnce(GetInfoValueByName("ShowOnce"));
            myLog.WriteLogLine("Imported ShowOnce.");

            IResourceEntry soundResource = GetSound(GetInfoValueByName("Sound"), myLog);
            if (soundResource != null)
            {
                newCon.Sound = soundResource;
            }
            myLog.WriteLogLine("Imported Sound.");

            newCon.Speaker = GetInfoValueByName("Speaker");
            myLog.WriteLogLine("Imported Speaker.");

            newCon.SpeakerAppearance = GetSpeakerAppearance(GetInfoObjectByName("SpeakerAppearance"));
            myLog.WriteLogLine("Imported SpeakerAppearance.");

            newCon.SpeakerGender = GetSpeakerGender(GetInfoValueByName("SpeakerGender"));
            myLog.WriteLogLine("Imported SpeakerGender.");

            int variation;
            if (int.TryParse(GetInfoValueByName("SpeakerHeadVariation"), out variation))
            {
                newCon.SpeakerHeadVariation = variation;
            }
            myLog.WriteLogLine("Imported SpeakerHeadVariation.");

            IResourceEntry tgaResource = GetTGAToDisplay(GetInfoValueByName("TGAToDisplay"), myLog);
            if (tgaResource != null)
            {
                newCon.TGAToDisplay = tgaResource;
            }
            myLog.WriteLogLine("Imported TGAToDisplay.");

            newCon.Type = GetConnectorType(GetInfoValueByName("Type"));
            myLog.WriteLogLine("Set Type.");

            newCon.VONeedsToBeRecorded = ConvertStringToBool(GetInfoValueByName("VONeedsToBeRecorded"));
            myLog.WriteLogLine("Set VONeedsToBeRecorded");

            string text = this.conversationText;
            //text = text.Replace(Environment.NewLine, "");
            newCon.Text.SetString(text, BWLanguages.BWLanguage.English, BWLanguages.Gender.Male);
            myLog.WriteLogLine("Set text.");
        }

        public void SetMyConversationForSubnodes(Conversation conv)
        {
            myConversation = conv;
            foreach (ContentNode subNode in subNodes)
            {
                subNode.SetMyConversationForSubnodes(conv);
            }
        }
    }
}
