using System;
using System.Collections.Generic;
using System.Collections;
using System.Text;
using System.Xml;
using System.Xml.Serialization;
using System.IO;
using System.Windows.Forms;
using NWN2Toolset.NWN2.Data;
using NWN2Toolset.NWN2.Data.TypedCollections;
using NWN2Toolset.NWN2.Data.ConversationData;
using NWN2Toolset.NWN2.Views;
using GlacialComponents.Controls.GlacialTreeList;
using ManagedElectron;
using OEIShared.Utils;
using OEIShared.IO;
using OEIShared.IO.TwoDA;
using NWN2Toolset.NWN2.Data.Templates;

namespace XmlExporter
{
    [Serializable]
    public class Conversation
    {
        [XmlArrayItem("ContentNode")]
        public List<ContentNode> subNodes = new List<ContentNode>();

        [XmlIgnore]
        public int maxIdNum = 1;

        [XmlIgnore]
        public Log log;

        [XmlArrayItem("Info")]
        public List<Info> additionalData = new List<Info>();

        [XmlArrayItem("InfoObject")]
        public List<InfoObject> infoObjects = new List<InfoObject>();

        [XmlElement]
        public int VersionNumber = 1;

        private Conversation()
        {
        }

        public static Conversation GetConversation(string xmlFileName, Log myLog)
        {
            Conversation toReturn = null;
            XmlSerializer serializer = new XmlSerializer(typeof(Conversation));
            FileStream myFileStream = null;
            try
            {
                myFileStream = new FileStream(xmlFileName, FileMode.Open);
                toReturn = (Conversation)serializer.Deserialize(myFileStream);
            }
            catch (Exception ex)
            {
                string strErr = String.Format("Unable to open xml file. Error:\n{0}", ex.Message);
                MessageBox.Show(strErr, "Open File Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
            finally
            {
                if (myFileStream != null)
                {
                    myFileStream.Close();
                }
                SortConversation(toReturn);
            }
            if (toReturn != null)
            {
                toReturn.log = myLog;
            }
            return toReturn;
        }

        public string GetInfoValueByName(string name)
        {
            foreach (Info info in additionalData)
            {
                if (info.variableName == name)
                {
                    return info.variableValue;
                }
            }
            return "";
        }

        public InfoObject GetInfoObjectByName(string name)
        {
            foreach (InfoObject info in infoObjects)
            {
                if (info.objectName == name)
                    return info;
            }
            return new InfoObject();
        }

        public static Info SetAbortConversationScript(IResourceEntry abortScript)
        {
            Info abortInfo = new Info();
            if (abortScript.ResRef != null)
            {
                abortInfo.variableName = "AbortConversationScript";
                abortInfo.variableValue = abortScript.ResRef.ToString();
                return abortInfo;
            }
            return null;
        }

        public static InfoObject SetDefaultSpeakerAppearance(TwoDAReference appearance)
        {
            if (appearance == null)
            {
                return null;
            }
            InfoObject appearanceObject = new InfoObject("DefaultSpeakerAppearance");
            appearanceObject.AddItem("TwoDAName", appearance.TwoDAName);
            appearanceObject.AddItem("ColumnName", appearance.ColumnName);
            appearanceObject.AddItem("IsStringRef", appearance.IsStringRef.ToString());
            appearanceObject.AddItem("Row", appearance.Row.ToString());

            return appearanceObject;
        }

        public static Info SetEndConversationScript(IResourceEntry endScript)
        {
            Info endInfo = new Info();
            if (endScript.ResRef != null)
            {
                endInfo.variableName = "EndConversationScript";
                endInfo.variableValue = endScript.ResRef.ToString();
                return endInfo;
            }
            return null;
        }
        
        public static Conversation NWN2ToConversation(NWN2GameConversation conv, Log myLog)
        {
            Conversation toReturn = new Conversation();

            Info abortInfo = SetAbortConversationScript(conv.AbortConversationScript);
            if (abortInfo != null)
            {
                toReturn.additionalData.Add(abortInfo);
            }
            myLog.WriteLogLine("set AbortConversationScript");

            toReturn.additionalData.Add(new Info("Comments", conv.Comments));
            myLog.WriteLogLine("set Comments for conversation");

            InfoObject defaultAppearance = SetDefaultSpeakerAppearance(conv.DefaultSpeakerAppearance);
            if (defaultAppearance != null)
            {
                toReturn.infoObjects.Add(defaultAppearance);
            }
            myLog.WriteLogLine("set DefaultSpeakerAppearance");

            toReturn.additionalData.Add(new Info("DefaultSpeakerGender", conv.DefaultSpeakerGender.ToString()));
            myLog.WriteLogLine("set DefaultSpeakerGender");

            toReturn.additionalData.Add(new Info("DefaultSpeakerHeadVariation", conv.DefaultSpeakerHeadVariation.ToString()));
            myLog.WriteLogLine("set DefaultSpeakerHeadVariation");

            toReturn.additionalData.Add(new Info("DelayEntry", conv.DelayEntry.ToString()));
            myLog.WriteLogLine("set DelayEntry");

            toReturn.additionalData.Add(new Info("DelayReply", conv.DelayReply.ToString()));
            myLog.WriteLogLine("set DelayReply");

            Info endInfo = SetEndConversationScript(conv.EndConversationScript);
            if (endInfo != null)
            {
                toReturn.additionalData.Add(endInfo);
            }
            myLog.WriteLogLine("set EndConversationScript");

            toReturn.additionalData.Add(new Info("MultiplayerCutscene", conv.MultiplayerCutscene.ToString()));
            myLog.WriteLogLine("set MultiplayerCutscene");

            toReturn.additionalData.Add(new Info("NWN1StyleDialogue", conv.NWN1StyleDialogue.ToString()));
            myLog.WriteLogLine("set NWN1StyleDialogue");

            toReturn.additionalData.Add(new Info("PreventZoom", conv.PreventZoom.ToString()));
            myLog.WriteLogLine("set PreventZoom");

            toReturn.additionalData.Add(new Info("UseDefaultSpeakerAppearance", conv.UseDefaultSpeakerAppearance.ToString()));
            myLog.WriteLogLine("set UseDefaultSpeakerAppearance");

            toReturn.additionalData.Add(new Info("VOCharacterName", conv.VOCharacterName));
            myLog.WriteLogLine("set VOCharacterName");
            
            return toReturn;
        }

        public static void SortList(List<ContentNode> thisList)
        {
            int changes = 1;
            int i;
            ContentNode tempNode;
            // sorts lists of ContentNodes according to the "orderNum" value of each node
            while (changes > 0)
            {
                changes = 0;
                for (i = 1; i < thisList.Count; i++)
                {
                    if (thisList[i-1].orderNum > thisList[i].orderNum)
                    {
                        changes++;
                        tempNode = thisList[i-1];
                        thisList[i-1] = thisList[i];
                        thisList[i] = tempNode;
                    }
                }
            }
        }

        public static void SortSubNodes(ContentNode myNode)
        {
            foreach (ContentNode subNode in myNode.subNodes)
            {
                SortSubNodes(subNode);
            }
            SortList(myNode.subNodes);
        }

        public static void SortConversation(Conversation toSort)
        {
            int i;
            if (toSort.subNodes.Count > 0)
            {
                for (i = 0; i < toSort.subNodes.Count; i++)
                {
                    SortSubNodes(toSort.subNodes[i]);
                }
            }
        }

        public void SaveContentConversation(string fileName)
        {
            StreamWriter writer = null;
            try
            {
                XmlSerializer ser = new XmlSerializer(typeof(Conversation));
                writer = new StreamWriter(fileName);
                ser.Serialize(writer, this);
            }
            catch (Exception ex)
            {
                string strErr = "Unable to save conversation. Error: " + ex.Message;
                MessageBox.Show(strErr, "Save File Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
            finally
            {
                if (writer != null)
                {
                    writer.Close();
                }
                writer = null;
            }
        }

        public ContentNode GetContentNodeById(int idNum)
        {
            ContentNode tempNode = null;
            foreach (ContentNode subNode in subNodes)
            {
                tempNode = subNode.SearchContentNodeById(idNum);
                if (tempNode != null)
                {
                    return tempNode;
                }
            }
            return null;
        }

        public string GetTextById(int idNum)
        {
            ContentNode tempNode = GetContentNodeById(idNum);
            return tempNode.conversationText;
        }

        public void AddNodeToRoot(ContentNode contentNode)
        {
            subNodes.Add(contentNode);
        }

        public void RemoveNodeFromRoot(ContentNode contentNode)
        {
            subNodes.Remove(contentNode);
        }

        public void ConnectAllLinks()
        {
            foreach (ContentNode node in subNodes)
            {
                node.ConnectLinks();
            }
        }

        public ContentNode FindNodeByLineGuid(string lineGuid)
        {
            ContentNode tempNode = null;
            foreach (ContentNode node in subNodes)
            {
                tempNode = node.FindNodeByLineGuid(lineGuid);
                if (tempNode != null)
                {
                    return tempNode;
                }
            }
            return null;
        }

        public NWN2ConversationLine GetNWN2LineByID(int idNum)
        {
            ContentNode foundNode = GetContentNodeById(idNum);
            if (foundNode != null && !foundNode.isLink)
            {
                if (foundNode.myConnector == null)
                {
                    MessageBox.Show("node does not have connector");
                    MessageBox.Show("The node matching idNum " + idNum.ToString() + " has the text " + foundNode.conversationText + ".");
                }
                if (foundNode.myConnector.Line == null)
                {
                    MessageBox.Show("node's connector does not have a line");
                }
                return foundNode.myConnector.Line;
            }
            else
            {
                return null;
            }
        }

        public static IResourceEntry GetResource(string name, string type, Log log)
        {
            ushort resourceType = BWResourceTypes.GetResourceType(type);
            IResourceEntry resource = null;
            OEIResRef resourceRef = new OEIResRef(name);
            log.WriteLogLine("Looking for resource of type '" + type + "' with name '" + name + "'");
            foreach (ResourceRepository repo in ResourceManager.Instance.Repositories)
            {
                resource = repo.FindResource(resourceRef, resourceType);
                if (resource != null)
                {
                    return resource;
                }
            }
            log.WriteLogLine("Script not found in any repositories.");
            return null;
        }

        public static bool ConvertStringToBoolean(string text)
        {
            if (text == "True" || text == "true")
            {
                return true;
            }
            else
            {
                return false;
            }
        }

        public static bool ArrayContains(string[] array, string contains)
        {
            foreach (string content in array)
            {
                if (content == contains)
                {
                    return true;
                }
            }
            return false;
        }

        public static TwoDAReference GetDefaultSpeakerAppearance(InfoObject speaker)
        {
            TwoDAReference appearance;
            if (speaker != null)
            {
                if (speaker.GetInfoByName("TwoDAName") != "" && speaker.GetInfoByName("ColumnName") != "" && speaker.GetInfoByName("IsStringRef") != "" && speaker.GetInfoByName("Row") != "")
                {
                    appearance = new TwoDAReference(speaker.GetInfoByName("TwoDAName"), speaker.GetInfoByName("ColumnName"), ConvertStringToBoolean(speaker.GetInfoByName("IsStringRef")), int.Parse(speaker.GetInfoByName("Row")));
                }
                else
                {
                    appearance = new TwoDAReference("Appearance", "STRING_REF", true, 0);
                }
            }
            else
            {
                appearance = new TwoDAReference("Appearance", "STRING_REF", true, 0);
            }

            return appearance;
        }

        public static CreatureGender GetDefaultSpeakerGender(string gender)
        {
            if (ArrayContains(Enum.GetNames(typeof(CreatureGender)), gender))
            {
                return (CreatureGender)Enum.Parse(typeof(CreatureGender), gender, false);
            }
            return CreatureGender.Male;
        }

        public void ConversationToNWN2(string conversationName)
        {
            string filePath = NWN2Toolset.NWN2ToolsetMainForm.App.Module.FileName;
            filePath = XmlExporter.GetFilePath(filePath);
            NWN2GameConversation conv = new NWN2GameConversation(conversationName, NWN2Toolset.NWN2ToolsetMainForm.App.Module.TempDirectory, NWN2Toolset.NWN2ToolsetMainForm.App.Module.Repository);

            NWN2ConversationConnector tempConnector;
            NWN2ConversationLine tempLine;

            conv.Demand();

            IResourceEntry abortResource = GetResource(GetInfoValueByName("AbortConversationScript"), "NSS", log);
            if (abortResource != null)
            {
                conv.AbortConversationScript = abortResource;
            }
            log.WriteLogLine("Set AbortConversationScript");

            conv.Comments = GetInfoValueByName("Comments");
            log.WriteLogLine("Set Comments");

            TwoDAReference defaultAppearance = GetDefaultSpeakerAppearance(GetInfoObjectByName("DefaultSpeakerAppearance"));
            if (defaultAppearance != null)
            {
                conv.DefaultSpeakerAppearance = defaultAppearance;
            }
            log.WriteLogLine("Set DefaultSpeakerAppearance");

            conv.DefaultSpeakerGender = GetDefaultSpeakerGender(GetInfoValueByName("DefaultSpeakerGender"));
            log.WriteLogLine("Set DefaultSpeakerGender");

            int variation;
            if (int.TryParse(GetInfoValueByName("DefaultSpeakerHeadVariation"), out variation))
            {
                conv.DefaultSpeakerHeadVariation = variation;
            }
            log.WriteLogLine("Set DefaultSpeakerHeadVariation");

            if (GetInfoValueByName("DelayEntry") != "")
            {
                conv.DelayEntry = float.Parse(GetInfoValueByName("DelayEntry"));
            }
            log.WriteLogLine("Set DelayEntry");

            if (GetInfoValueByName("DelayReply") != "")
            {
                conv.DelayReply = float.Parse(GetInfoValueByName("DelayReply"));
            }
            log.WriteLogLine("Set DelayReply");

            IResourceEntry endResource = GetResource(GetInfoValueByName("EndConversationScript"), "NSS", log);
            if (endResource != null)
            {
                conv.EndConversationScript = endResource;
            }
            log.WriteLogLine("Set EndConversationScript");

            conv.MultiplayerCutscene = ConvertStringToBoolean(GetInfoValueByName("MultiplayerCutscene"));
            log.WriteLogLine("Set MultiplayerCutscene");

            conv.NWN1StyleDialogue = ConvertStringToBoolean(GetInfoValueByName("NWN1StyleDialogue"));
            log.WriteLogLine("Set NWN1StyleDialogue");

            conv.PreventZoom = ConvertStringToBoolean(GetInfoValueByName("PreventZoom"));
            log.WriteLogLine("Set PreventZoom");

            conv.UseDefaultSpeakerAppearance = ConvertStringToBoolean(GetInfoValueByName("UseDefaultSpeakerAppearance"));
            log.WriteLogLine("Set UseDefaultSpeakerAppearance");

            conv.VOCharacterName = GetInfoValueByName("VOCharacterName");
            log.WriteLogLine("Set VOCharacterName");

            foreach (ContentNode subNode in this.subNodes)
            {
                if (!subNode.isLink)
                {
                    tempConnector = conv.InsertChild(null);
                    subNode.PopulateNWN2ConversationConnector(tempConnector, log);
                    subNode.AddMyChildrenToMyNWN2ConversationConnector(conv, log);
                }
            }

            foreach (ContentNode subNode in this.subNodes)
            {
                if (subNode.isLink)
                {
                    tempLine = this.GetNWN2LineByID(subNode.linkTo);
                    if (tempLine != null)
                    {
                        conv.InsertLink(null, tempLine);
                    }
                }
                else
                {
                    subNode.AddMyChildLinksToMyNWN2ConversationConnector(this, conv, log);
                }
            }

            conv = (NWN2GameConversation)NWN2Toolset.NWN2ToolsetMainForm.App.Module.AddResource(conv);
            NWN2Toolset.NWN2ToolsetMainForm.App.ShowResource(conv);
        }

        public bool IsSubnode(ContentNode node)
        {
            foreach (ContentNode subNode in subNodes)
            {
                if (subNode == node)
                {
                    return true;
                }
            }
            return false;
        }

        public void SetMyConversationForSubnodes()
        {
            foreach (ContentNode subNode in subNodes)
            {
                subNode.SetMyConversationForSubnodes(this);
            }
        }
    }
}
